/**
 * Code for Blockly Editor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

'use strict';

/**
 * Create a namespace for the application.
 */
var Code = {};

/**
 * Lookup for names of supported languages.  Keys should be in ISO 639 format.
 */
Code.LANGUAGE_NAME = {
  'ar': 'العربية',
  'be-tarask': 'Taraškievica',
  'br': 'Brezhoneg',
  'ca': 'Català',
  'cs': 'Česky',
  'da': 'Dansk',
  'de': 'Deutsch',
  'el': 'Ελληνικά',
  'en': 'English',
  'es': 'Español',
  'et': 'Eesti',
  'fa': 'فارسی',
  'fr': 'Français',
  'he': 'עברית',
  'hrx': 'Hunsrik',
  'hu': 'Magyar',
  'ia': 'Interlingua',
  'is': 'Íslenska',
  'it': 'Italiano',
  'ja': '日本語',
  'kab': 'Kabyle',
  'ko': '한국어',
  'mk': 'Македонски',
  'ms': 'Bahasa Melayu',
  'nb': 'Norsk Bokmål',
  'nl': 'Nederlands, Vlaams',
  'oc': 'Lenga d\'òc',
  'pl': 'Polski',
  'pms': 'Piemontèis',
  'pt-br': 'Português Brasileiro',
  'ro': 'Română',
  'ru': 'Русский',
  'sc': 'Sardu',
  'sk': 'Slovenčina',
  'sr': 'Српски',
  'sv': 'Svenska',
  'ta': 'தமிழ்',
  'th': 'ภาษาไทย',
  'tlh': 'tlhIngan Hol',
  'tr': 'Türkçe',
  'uk': 'Українська',
  'vi': 'Tiếng Việt',
  'zh-hans': '简体中文',
  'zh-hant': '正體中文'
};

/**
 * List of RTL languages.
 */
Code.LANGUAGE_RTL = ['ar', 'fa', 'he', 'lki'];

/**
 * Blockly's main workspace.
 * @type {Blockly.WorkspaceSvg}
 */
Code.workspace = null;

/**
 * Extracts a parameter from the URL.
 * If the parameter is absent default_value is returned.
 * @param {string} name The name of the parameter.
 * @param {string} defaultValue Value to return if parameter not found.
 * @return {string} The parameter value or the default value if not found.
 */
Code.getStringParamFromUrl = function(name, defaultValue) {
  var val = location.search.match(new RegExp('[?&]' + name + '=([^&]+)'));
  return val ? decodeURIComponent(val[1].replace(/\+/g, '+')) : defaultValue;
};

/**
 * Get the language of this user from the URL.
 * @return {string} User's language.
 */
Code.getLang = function() {
  var lang = Code.getStringParamFromUrl('lang', '');
  if (Code.LANGUAGE_NAME[lang] === undefined) {
    // Default to English.
    lang = 'en';
  }
  return lang;
};

/**
 * Is the current language (Code.LANG) an RTL language?
 * @return {boolean} True if RTL, false if LTR.
 */
Code.isRtl = function() {
  return Code.LANGUAGE_RTL.indexOf(Code.LANG) != -1;
};

/**
 * Load blocks saved on App Engine Storage or in session/local storage.
 * @param {string} defaultXml Text representation of default blocks.
 */
Code.loadBlocks = function(defaultXml) {
  try {
    var loadOnce = window.sessionStorage.loadOnceBlocks;
  } catch(e) {
    // Firefox sometimes throws a SecurityError when accessing sessionStorage.
    // Restarting Firefox fixes this, so it looks like a bug.
    var loadOnce = null;
  }
  if ('BlocklyStorage' in window) { //&& window.location.hash.length > 1
    // An href with #key trigers an AJAX call to retrieve saved blocks.
    
    BlocklyStorage.retrieveXml(window.location.hash.substring(1));
  } else if (loadOnce) {
    // Language switching stores the blocks during the reload.
    delete window.sessionStorage.loadOnceBlocks;
    var xml = Blockly.Xml.textToDom(loadOnce);
    Blockly.Xml.domToWorkspace(xml, Code.workspace);
  } else if (defaultXml) {
    // Load the editor with default starting blocks.
    var xml = Blockly.Xml.textToDom(defaultXml);
    Blockly.Xml.domToWorkspace(xml, Code.workspace);
  } else if ('BlocklyStorage' in window) {
    // Restore saved blocks in a separate thread so that subsequent
    // initialization is not affected from a failed load.
    window.setTimeout(BlocklyStorage.restoreBlocks, 0);
  }
};

Code.loadBlockTypes=function() {
    Blockly.Blocks['wm_comment'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('comment ')
            .appendField(new Blockly.FieldTextInput('enter a comment here'),
                     'VAR');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(20);
        this.setTooltip("Add a comment ");
      }
    };
    Blockly.PHP['wm_comment'] = function(block) {
        var value_role_id =block.getFieldValue('VAR');
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = '// '+value_role_id + '\r\n';
      return code;
    };
    
    // mb_get_val_by_role
    Blockly.Blocks['wm_mb_get_val_by_role'] = {
      init: function() {
        this.appendValueInput("ROLE_ID")
            .setCheck("Number")
            .appendField('value of role ');
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(65);
     this.setTooltip("Get the value of a role linked to a device");
     this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mb_get_val_by_role");
      }
    };
    Blockly.PHP['wm_mb_get_val_by_role'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'ROLE_ID', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'mb_get_val_by_role('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };


    Blockly.Blocks['wm_sizeof'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('size of ');
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
     this.setTooltip("Get the length of an array");
      }
    };
    Blockly.PHP['wm_sizeof'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'sizeof('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_is_array'] = {
      init: function() {
       this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is variable');
        this.appendDummyInput()
            .appendField('an array?');
        this.setInputsInline(true); 
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
        this.setTooltip("Check if a variable is an array");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/is_array");
      }
    };
    Blockly.PHP['wm_is_array'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'is_array('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_is_float'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is variable');
        this.appendDummyInput()
            .appendField('a float?');
        this.setInputsInline(true);        
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
        this.setTooltip("Check if a variable is a float");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/is_float");
      }
    };
    Blockly.PHP['wm_is_float'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'is_float('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_is_int'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is variable');
        this.appendDummyInput()
            .appendField('an int?');
        this.setInputsInline(true);        
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
        this.setTooltip("Check if a variable is an int");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/is_int");
      }
    };
    Blockly.PHP['wm_is_int'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'is_int('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_is_numeric'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is variable');
        this.appendDummyInput()
            .appendField('numeric?');
        this.setInputsInline(true);        
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
        this.setTooltip("Check if a variable is numeric");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/is_numeric");
      }
    };
    Blockly.PHP['wm_is_numeric'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'is_numeric('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_is_string'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is variable');
        this.appendDummyInput()
            .appendField('a string?');
        this.setInputsInline(true);        
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
        this.setTooltip("Check if a variable is a string");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/is_string");
      }
    };
    Blockly.PHP['wm_is_string'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'is_string('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_is_set'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is set ');
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(260);
     this.setTooltip("Check if a variable is set");
      }
    };
    Blockly.PHP['wm_is_set'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = 'isset('+value_role_id+')';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_adc_read'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("Number")
            .appendField('get adc value of channel');
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(65);
        this.setTooltip("Read an onboard ADC channel");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/adc_read");
      }
    };
    Blockly.PHP['wm_adc_read'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'adc_read('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    
    Blockly.Blocks['wm_array_keys'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('get keys for array ');
        this.setOutput(true, null);
        this.setColour(260);
        this.setTooltip("Return keys for an array that has key/value pairs");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/array_keys");
      }
    };
    Blockly.PHP['wm_array_keys'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'array_keys('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    
    Blockly.Blocks['wm_base64_decode'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('base64_decode ');
        this.setOutput(true, ["String"]);
        this.setColour(65);
        this.setTooltip("Decode a base64-encoded string");
      }
    };
    Blockly.PHP['wm_base64_decode'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'base64_decode('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_base64_encode'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('base64 encode ');
        this.setOutput(true, ["String"]);
        this.setColour(65);
        this.setTooltip("Return the base64-encoded version of a string");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/base65_encode");
      }
    };
    Blockly.PHP['wm_base64_encode'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'base64_encode('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    Blockly.Blocks['wm_call_user_func'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('call user-defined function ');
        this.setOutput(true, ["Number","String"]);
        this.setColour(65);
        this.setTooltip("Call a user defined function by name");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/call_user_func");
      }
    };
    Blockly.PHP['wm_call_user_func'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'call_user_func("'+value_role_id+'")';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    Blockly.Blocks['wm_charat'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('get character in ');
        this.appendValueInput("IDX")
            .setCheck("Number")
            .appendField('at index ');
        this.setInputsInline(true);    
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Return the ASCII code of a charater in a string");
        
      }
    };
    Blockly.PHP['wm_charat'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var idx_id = Blockly.PHP.valueToCode(block, 'IDX', Blockly.PHP.ORDER_ATOMIC);
      var code = 'charat('+value_role_id+','+idx_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_strftime'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('format time using format ');
        this.appendValueInput("IDX")
            .setCheck("Number")
            .appendField('from timestamp ');
        this.setInputsInline(true);    
        this.setOutput(true, ["String"]);
        this.setColour(65);
        this.setTooltip("Return a formatted unix timestamp");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/strftime");
      }
    };
    Blockly.PHP['wm_strftime'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var idx_id = Blockly.PHP.valueToCode(block, 'IDX', Blockly.PHP.ORDER_ATOMIC);
      var code = 'strftime('+value_role_id+','+idx_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_number_format'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('number format value ');
        this.appendValueInput("IDX")
            .setCheck("Number")
            .appendField('with precision ');
        this.setInputsInline(true);    
        this.setOutput(true, ["String"]);
        this.setColour(65);
        this.setTooltip("Return the string of a number with fixed precision");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/number_format");
      }
    };
    Blockly.PHP['wm_number_format'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var idx_id = Blockly.PHP.valueToCode(block, 'IDX', Blockly.PHP.ORDER_ATOMIC);
      var code = 'number_format('+value_role_id+','+idx_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mail'] = {
      init: function() {
        this.appendValueInput("REC")
            .setCheck("String")
            .appendField('send an email to ');
        this.appendValueInput("SUB")
            .setCheck("String")
            .appendField('with subject ');
        this.appendValueInput("BODY")
            .setCheck("String")
            .appendField('and body ');
            
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Send an email");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mail");
      }
    };
    Blockly.PHP['wm_mail'] = function(block) {
      var rec = Blockly.PHP.valueToCode(block, 'REC', Blockly.PHP.ORDER_ATOMIC);
      var sub = Blockly.PHP.valueToCode(block, 'SUB', Blockly.PHP.ORDER_ATOMIC);
      var body = Blockly.PHP.valueToCode(block, 'BODY', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mail('+rec+',' + sub + ',' + body + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_add_dev'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('add a modbus device with id ');
        this.appendValueInput("TYPE")
            .setCheck(null)
            .appendField('and driver type id');
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('and name ');
        this.appendValueInput("POLL")
            .setCheck("Number")
            .appendField('and poll interval (in ms) ');
        this.appendDummyInput()
            .appendField('and status ')
            .appendField(new Blockly.FieldDropdown([
                       ['disabled','0'],
                       ['enabled','1']
                     ]),
                     "STATUS");
        this.appendDummyInput()
            .appendField('on bus ')
            .appendField(new Blockly.FieldDropdown([
                       ['RS-485','0'],
                       ['RS-485 2','1'],
                       ['TCP Channel 1','2'],
                       ['TCP Channel 2','3'],
                       ['TCP Channel 3','4'],
                       ['TCP Channel 1','5'],
                       ['TCP Channel 4','6'],
                       ['TCP Channel 5','7'],
                       ['TCP Channel 6','8'],
                       ['TCP Channel 7','9'],
                       ['TCP Channel 8','10'],
                       ['TCP Channel 9','11'],
                       ['TCP Channel 10','12'],
                       ['Virtual','100']
                     ]),
                     "BUS");           

        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Add a device");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_add_dev");
      }
    };
    Blockly.PHP['wm_mb_add_dev'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var type = Blockly.PHP.valueToCode(block, 'TYPE', Blockly.PHP.ORDER_ATOMIC);
      var name = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      var poll = Blockly.PHP.valueToCode(block, 'POLL', Blockly.PHP.ORDER_ATOMIC);
      var status =block.getFieldValue('STATUS');
      var bus =block.getFieldValue('BUS');
      var code = 'mb_add_dev('+dev_id+',' +
                        type + ',' + 
                        name + ',' +
                        poll + ',' + 
                        status + ',' + 
                        bus + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_mb_delete_dev'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('delete modbus device with id ');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Delete a modbus device");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_delete_dev");
      }
    };
    Blockly.PHP['wm_mb_delete_dev'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_delete_dev('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_get_dev_by_id'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('get array of values for modbus device id ');
        
        this.appendDummyInput()
            .appendField('on bus ')
            .appendField(new Blockly.FieldDropdown([
                       ['RS-485','0'],
                       ['RS-485 2','1'],
                       ['TCP Channel 1','2'],
                       ['TCP Channel 2','3'],
                       ['TCP Channel 3','4'],
                       ['TCP Channel 1','5'],
                       ['TCP Channel 4','6'],
                       ['TCP Channel 5','7'],
                       ['TCP Channel 6','8'],
                       ['TCP Channel 7','9'],
                       ['TCP Channel 8','10'],
                       ['TCP Channel 9','11'],
                       ['TCP Channel 10','12'],
                       ['Virtual','100']
                     ]),
                     "BUS");             
        this.setOutput(true, ["Array"]);
        this.setInputsInline(true);    
        this.setColour(65);
        this.setTooltip("Get modbus device values by id");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_dev_by_id");
      }
    };
    Blockly.PHP['wm_mb_get_dev_by_id'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var bus  =block.getFieldValue('BUS');
      var code = 'mb_get_dev_by_id('+dev_id+',' + bus+
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_mb_get_dev_by_index'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('get array of values for modbus device index ');
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Get modbus device values by index");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_dev_by_index");
      }
    };
    Blockly.PHP['wm_mb_get_dev_by_index'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_get_dev_by_index('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_mb_get_dev_by_name'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Text")
            .appendField('get array of values for modbus device with name ');
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Get modbus device values by hame");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_dev_by_name");
      }
    };
    Blockly.PHP['wm_mb_get_dev_by_name'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_get_dev_by_name('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_get_dev_info'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('get device type info for device type id ');
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Get modbus device type info as array");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_dev_info");
      }
    };
    Blockly.PHP['wm_mb_get_dev_info'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_get_dev_info('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_mb_get_role_array'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('get array of all roles');
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Get array of all roles defined");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_role_array");
      }
    };
    Blockly.PHP['wm_mb_get_role_array'] = function(block) {
      var code = 'mb_get_role_array()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_get_status_by_role'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Number")
            .appendField('get the status of a device that is linked to role id ');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get device status linked to role id");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_get_status_by_role");
      }
    };
    Blockly.PHP['wm_mb_get_status_by_role'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_get_status_by_role('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_num_devices'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('number of devices');
        this.setOutput(true, ["Nynber"]);
        this.setColour(65);
        this.setTooltip("Get number of devices");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_num_devices");
      }
    };
    Blockly.PHP['wm_mb_num_devices'] = function(block) {
      var code = 'mb_num_devices()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_queue_command'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Array")
            .appendField('queue a modbus command in array ');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Queue a modbus command");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_queue_command");
      }
    };
    Blockly.PHP['wm_mb_queue_command'] = function(block) {
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_queue_command('+dev_id+',' +
                         ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_send_command'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck("Array")
            .appendField('send a modbus command ');
         this.appendDummyInput()
            .appendField('on bus ')
            .appendField(new Blockly.FieldDropdown([
                       ['RS-485','0'],
                       ['RS-485 2','1'],
                       ['TCP Channel 1','2'],
                       ['TCP Channel 2','3'],
                       ['TCP Channel 3','4'],
                       ['TCP Channel 1','5'],
                       ['TCP Channel 4','6'],
                       ['TCP Channel 5','7'],
                       ['TCP Channel 6','8'],
                       ['TCP Channel 7','9'],
                       ['TCP Channel 8','10'],
                       ['TCP Channel 9','11'],
                       ['TCP Channel 10','12'],
                       ['Virtual','100']
                     ]),
                     "BUS");             
        this.setInputsInline(true);    
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Send a modbus command");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_send_command");
      }
    };
    Blockly.PHP['wm_mb_send_command'] = function(block) {
      var bus  =block.getFieldValue('BUS');
      var dev_id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_send_command('+bus+','+dev_id+',' + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_mb_start_scan'] = {
      init: function() {
           this.appendDummyInput()
            .appendField('start a scan for new Wattmon devices');
        this.appendValueInput("START")
            .setCheck("Number")
            .appendField('starting at ID ');
        this.appendValueInput("END")
            .setCheck("Number")
            .appendField('ending at ID ');
            
        this.appendDummyInput()    
            .appendField('on bus ')
            .appendField(new Blockly.FieldDropdown([
                       ['RS-485','0'],
                       ['RS-485 2','1'],
                       ['TCP Channel 1','2'],
                       ['TCP Channel 2','3'],
                       ['TCP Channel 3','4'],
                       ['TCP Channel 1','5'],
                       ['TCP Channel 4','6'],
                       ['TCP Channel 5','7'],
                       ['TCP Channel 6','8'],
                       ['TCP Channel 7','9'],
                       ['TCP Channel 8','10'],
                       ['TCP Channel 9','11'],
                       ['TCP Channel 10','12'],
                       ['Virtual','100']
                     ]),
                     "BUS");             
            
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Start a scan for new Wattmon devices");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mb_start_scan");
      }
    };
    
    Blockly.PHP['wm_mb_start_scan'] = function(block) {
      var start = Blockly.PHP.valueToCode(block, 'START', Blockly.PHP.ORDER_ATOMIC);
      var end = Blockly.PHP.valueToCode(block, 'END', Blockly.PHP.ORDER_ATOMIC);
      var bus  =block.getFieldValue('BUS');
      var code = 'mb_start_scan(' + 
            start + ','+
            end + ','+
            bus + ');\n';
      return code;
    };
    
    
    Blockly.Blocks['wm_mb_scan_complete'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('scan complete');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get number of devices");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_scan_complete");
      }
    };
    Blockly.PHP['wm_mb_scan_complete'] = function(block) {
      var code = 'mb_scan_complete()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_scan_percent'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('scan percent');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get scan percent");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/mb_scan_percent");
      }
    };
    Blockly.PHP['wm_mb_scan_percent'] = function(block) {
      var code = 'wm_mb_scan_percent()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_md5'] = {
      init: function() {
        this.appendValueInput("DIR")
            .setCheck(null)
            .appendField('calculate MD5 checksum for string');
        this.setOutput(true, ["String"]);
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Calculate MD5 checksum for string");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/md5");
      }
    };
    Blockly.PHP['wm_md5'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'md5(' + value_role_id + ');\n';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_untar'] = {
      init: function() {
        this.appendValueInput("DIR")
            .setCheck(null)
            .appendField('untar file ');
        this.appendValueInput("VER")
            .setCheck("Number")
            .appendField('with verbosity ');
            
        this.setOutput(true, ["Number"]);
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Untar a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/untar");
      }
    };
    Blockly.PHP['wm_untar'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var value_role_id2 = Blockly.PHP.valueToCode(block, 'VER', Blockly.PHP.ORDER_ATOMIC);
      var code = 'untar(' + value_role_id + ',' + value_role_id2 +')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_sha1'] = {
      init: function() {
        this.appendValueInput("DIR")
            .setCheck(null)
            .appendField('calculate SHA1 checksum for string');
        this.setOutput(true, ["String"]);
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Calculate SHA1 checksum for string");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/sha1");
      }
    };
    Blockly.PHP['wm_sha1'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'sha1(' + value_role_id + ');\n';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_md5_file'] = {
      init: function() {
        this.appendValueInput("DIR")
            .setCheck("String")
            .appendField('calculate MD5 checksum for file');
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Calculate MD5 checksum for a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/md5_file");
      }
    };
    Blockly.PHP['wm_md5_file'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'md5_file(' + value_role_id + ');\n';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_microtime'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('system uptime in ms');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get system uptime in milliseconds");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/microtime");
      }
    };
    Blockly.PHP['wm_microtime'] = function(block) {
      var code = 'microtime()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_uptime'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('system uptime in ms');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get system uptime in milliseconds");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/uptime");
      }
    };
    Blockly.PHP['wm_uptime'] = function(block) {
      var code = 'uptime()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_time'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('system time as unix timestamp');
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Get system time");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/time");
      }
    };
    Blockly.PHP['wm_time'] = function(block) {
      var code = 'time()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
   Blockly.Blocks['wm_timefromfat'] = {
      init: function() {
        this.appendValueInput("NUM")
            .setCheck(null)
            .appendField('get unix timestamp from FAT timestamp');
        this.setOutput(true, ["Number"]);
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get unix timestamp from FAT time");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/timefromfat");
      }
    };
    Blockly.PHP['wm_timefromfat'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'timefromfat(' + value_role_id + ');\n';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mkdir'] = {
      init: function() {
            this.appendValueInput("DIR")
            .setCheck("String")
            .appendField('create directory');
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Create directory");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mkdir");
      }
    };
    Blockly.PHP['wm_mkdir'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mkdir(' + value_role_id + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
     Blockly.Blocks['wm_mktime'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('make timestamp from');
        this.appendValueInput("HOUR")
            .setCheck("Number")
            .appendField('hour');
        this.appendValueInput("MINUTE")
            .setCheck("Number")
            .appendField('minute');
        this.appendValueInput("SECOND")
            .setCheck("Number")
            .appendField('second');
        this.appendValueInput("YEAR")
            .setCheck("Number")
            .appendField('year');
        this.appendValueInput("MONTH")
            .setCheck("Number")
            .appendField('month');
        this.appendValueInput("DAY")
            .setCheck("Number")
            .appendField('day');
            
        this.setOutput(true, ["Number"]);
        this.setColour(65);
        this.setTooltip("Make a unix timestamp from variables");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mktime");
      }
    };
    Blockly.PHP['wm_mktime'] = function(block) {
      var hour = Blockly.PHP.valueToCode(block, 'HOUR', Blockly.PHP.ORDER_ATOMIC);
      var minute = Blockly.PHP.valueToCode(block, 'MINUTE', Blockly.PHP.ORDER_ATOMIC);
      var second = Blockly.PHP.valueToCode(block, 'SECOND', Blockly.PHP.ORDER_ATOMIC);
      var year = Blockly.PHP.valueToCode(block, 'YEAR', Blockly.PHP.ORDER_ATOMIC);
      var month = Blockly.PHP.valueToCode(block, 'MONTH', Blockly.PHP.ORDER_ATOMIC);
      var day = Blockly.PHP.valueToCode(block, 'DAY', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mktime(' + hour + ',' + minute + ','+second + ','+month + ','+day + ','+year + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mqtt_publish'] = {
      init: function() {
        this.appendValueInput("CHANNEL")
            .setCheck("String")
            .appendField('publish MQTT message to channel ');
        this.appendValueInput("CONTENT")
            .setCheck("String")
            .appendField('with content ');

        //this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(65);
        this.setTooltip("Publish MQTT message to channel");
      }
    };
    Blockly.PHP['wm_mqtt_publish'] = function(block) {
      var chan = Blockly.PHP.valueToCode(block, 'CHANNEL', Blockly.PHP.ORDER_ATOMIC);
      var content = Blockly.PHP.valueToCode(block, 'CONTENT', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mqtt_publish(' + chan + ','+content+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mqtt_subscribe'] = {
      init: function() {
        this.appendValueInput("CHANNEL")
            .setCheck("String")
            .appendField('subscribe to MQTT channel ');
        this.appendValueInput("CALLBACK")
            .setCheck("String")
            .appendField('with callback ');

        //this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(65);
        this.setTooltip("Subscribe to an MQTT channel");
      }
    };
    Blockly.PHP['wm_mqtt_subscribe'] = function(block) {
      var chan = Blockly.PHP.valueToCode(block, 'CHANNEL', Blockly.PHP.ORDER_ATOMIC);
      var callback = Blockly.PHP.valueToCode(block, 'CALLBACK', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mqtt_subscribe(' + chan + ','+callback+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ini_get'] = {
      init: function() {
        this.appendValueInput("KEY")
            .setCheck("String")
            .appendField('get value for key ');
        this.appendValueInput("SECTION")
            .setCheck("String")
            .appendField('in section ');
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('in file ');
        this.appendValueInput("DEF")
            .setCheck(null)
            .appendField('or use default value ');
            
        //this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(65);
        this.setTooltip("Get a value from an INI file");
      }
    };
    
   
    
    Blockly.PHP['wm_ini_get'] = function(block) {
      var key = Blockly.PHP.valueToCode(block, 'KEY', Blockly.PHP.ORDER_ATOMIC);
      var section = Blockly.PHP.valueToCode(block, 'SECTION', Blockly.PHP.ORDER_ATOMIC);
      var fn = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var def = Blockly.PHP.valueToCode(block, 'DEF', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ini_get('+fn+','+section+','+key+','+def+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ini_get_array'] = {
      init: function() {
        this.appendValueInput("SECTION")
            .setCheck("String")
            .appendField('get array of values in section ');
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('in file ');

        //this.setInputsInline(true);    
        this.setOutput(true, "Array");
        this.setColour(65);
        this.setTooltip("Get a value from an INI file");
      }
    };
    Blockly.PHP['wm_ini_get_array'] = function(block) {
      var section = Blockly.PHP.valueToCode(block, 'SECTION', Blockly.PHP.ORDER_ATOMIC);
      var fn = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ini_get_array('+fn+','+section+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ini_put_array'] = {
      init: function() {
        this.appendValueInput("ARR")
            .setCheck("Array")
            .appendField('save array ');
          
        this.appendValueInput("SECTION")
            .setCheck("String")
            .appendField('into section ');
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('in file ');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        //this.setInputsInline(true);    
        //this.setOutput(true, null);
        this.setColour(65);
        this.setTooltip("Get a value from an INI file");
      }
    };
    Blockly.PHP['wm_ini_put_array'] = function(block) {
      var arr = Blockly.PHP.valueToCode(block, 'ARR', Blockly.PHP.ORDER_ATOMIC);
      var section = Blockly.PHP.valueToCode(block, 'SECTION', Blockly.PHP.ORDER_ATOMIC);
      var fn = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ini_put_array('+fn+','+arr+','+section+');\n';
      return code;
    };
    
    Blockly.Blocks['wm_ini_set'] = {
      init: function() {
        this.appendValueInput("VAL")
            .setCheck(null)
            .appendField('set ini file value ');
        this.appendValueInput("KEY")
            .setCheck("String")
            .appendField('for key ');
        this.appendValueInput("SECTION")
            .setCheck("String")
            .appendField('in section ');
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('in file ');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        //this.setInputsInline(true);    
        //this.setOutput(true, null);
        this.setColour(65);
        this.setTooltip("Get a value from an INI file");
      }
    };
    Blockly.PHP['wm_ini_set'] = function(block) {
      var val = Blockly.PHP.valueToCode(block, 'VAL', Blockly.PHP.ORDER_ATOMIC);
      var key = Blockly.PHP.valueToCode(block, 'KEY', Blockly.PHP.ORDER_ATOMIC);
      var section = Blockly.PHP.valueToCode(block, 'SECTION', Blockly.PHP.ORDER_ATOMIC);
      var fn = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ini_set('+fn+','+section+','+key+','+val+');\n';
      return code;
    };
    
    Blockly.Blocks['wm_hash_hmac'] = {
      init: function() {
          this.appendDummyInput()
            .appendField('generate HMAC hash using algorithm ')
            .appendField(new Blockly.FieldDropdown([
                       ['MD5','md5'],
                       ['SHA1','sha1'],
                       ['SHA-224','sha224'],
                       ['SHA-256','sha256'],
                       ['SHA-384','sha384'],
                       ['SHA-512','sha512']
                     ]),
                     "ALGO")
            
            this.appendValueInput("DATA")
            .setCheck("String")
            .appendField(' for data ');
            
            this.appendValueInput("KEY")
            .setCheck("String")
            .appendField('using key ');
            
            
             
        this.setInputsInline(true);    
        this.setOutput(true, ["String"]);
        this.setColour(65);
        this.setTooltip("Return the HMAC for a string");
      }
    };
    Blockly.PHP['wm_hash_hmac'] = function(block) {
      var value_role_id =block.getFieldValue('ALGO');
      console.log(value_role_id);
      var data = Blockly.PHP.valueToCode(block, 'DATA', Blockly.PHP.ORDER_ATOMIC);
      var key = Blockly.PHP.valueToCode(block, 'KEY', Blockly.PHP.ORDER_ATOMIC);
      var code = 'hash_hmac("'+value_role_id+'",'+data+','+key+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_indexed_array'] = {
      init: function() {
          this.appendDummyInput()
            .appendField('Create an indexed array of type ')
            .appendField(new Blockly.FieldDropdown([
                       ['byte','1'],
                       ['short (16-bit)','2'],
                       ['int (32-bit)','3'],
                       ['float (32-bit)','4'],
                     ]),
                     "TYPE")
            
            this.appendValueInput("NUM")
            .setCheck("Number")
            .appendField(' with  ');
            
            this.appendDummyInput()
            .appendField('items ');
            
            
             
        this.setInputsInline(true);    
        this.setOutput(true, ["Array"]);
        this.setColour(65);
        this.setTooltip("Create an indexed array");
      }
    };
    Blockly.PHP['wm_indexed_array'] = function(block) {
      var value_role_id =block.getFieldValue('TYPE');
      var num = Blockly.PHP.valueToCode(block, 'NUM', Blockly.PHP.ORDER_ATOMIC);
      var code = 'indexed_array('+value_role_id+','+num+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mb_set_val_by_role'] = {
      init: function() {
        this.appendValueInput("ROLE_ID")
            .setCheck("Number")
            .appendField('set role');
        this.appendValueInput("VALUE")
            .setCheck(null)
            .appendField('to value');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set the value of a role linked to a device");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mb_set_val_by_role");
      }
    };
    
    Blockly.PHP['wm_mb_set_val_by_role'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'ROLE_ID', Blockly.PHP.ORDER_ATOMIC);
      var value_role_val = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_set_val_by_role(' + value_role_id + ', ' + value_role_val + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_mb_set_dev_var'] = {
      init: function() {
        this.appendValueInput("ID")
            .setCheck(null)
            .appendField('set variable for device ');
        this.appendValueInput("VARIABLE")
            .setCheck(null)
            .appendField('and variable name');
        this.appendValueInput("VALUE")
            .setCheck(null)
            .appendField('to value');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set a device variable value");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mb_set_dev_var");
      }
    };
    
    Blockly.PHP['wm_mb_set_dev_var'] = function(block) {
      var id = Blockly.PHP.valueToCode(block, 'ID', Blockly.PHP.ORDER_ATOMIC);
      var variable = Blockly.PHP.valueToCode(block, 'VARIABLE', Blockly.PHP.ORDER_ATOMIC);
      var value = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      var code = 'mb_set_dev_var(' + id + ', ' + variable + ', ' + value +');\n';
      return code;
    };
    
    Blockly.Blocks['wm_print_r'] = {
      init: function() {
        this.appendValueInput("ROLE_ID")
            .setCheck(null)
            .appendField('dump contents of array');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Dump the contents of the array for debugging");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/print_r");
      }
    };
    
    Blockly.PHP['wm_print_r'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'ROLE_ID', Blockly.PHP.ORDER_ATOMIC);
      var code = 'print_r(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_header'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('add to HTTP header value');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Add an HTTP header string");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/print_r");
      }
    };
    
    Blockly.PHP['wm_header'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'header(' + value_role_id  + ');\n';
      return code;
    };
    
    
     Blockly.Blocks['wm_sleep'] = {
      init: function() {
        this.appendValueInput("MS")
            .setCheck("Number")
            .appendField('sleep for');
        this.appendDummyInput()
            .appendField('milliseconds');    
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Sleep for specified time");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/sleep");
      }
    };
    
    Blockly.PHP['wm_sleep'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'MS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'sleep(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_setethpower'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("Number")
            .appendField('set power of ethernet module to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set power of ethernet module");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/setethpower");
      }
    };
    
    Blockly.PHP['wm_setethpower'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'setethpower(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_setusbpower'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("Number")
            .appendField('set power of USB module to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set power of USB module");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/setusbpower");
      }
    };
    
    Blockly.PHP['wm_setusbpower'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'setusbpower(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_setpriority'] = {
      init: function() {
        this.appendValueInput("NUM")
            .setCheck("Number")
            .appendField('set script execution priority to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set execution priority of script");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/setpriority");
      }
    };
    
    Blockly.PHP['wm_setpriority'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NUM', Blockly.PHP.ORDER_ATOMIC);
      var code = 'setpriority(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_settime'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("Number")
            .appendField('set system time to unix timestamp ');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set system time to unix timestamp");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/settime");
      }
    };
    
    Blockly.PHP['wm_settime'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'settime(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_set_search_path'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('set telnet search path to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set search path for telnet");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/set_search_path");
      }
    };
    
    Blockly.PHP['wm_set_search_path'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'set_search_path(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_reset'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('reset the device');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Reset the device");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/reset");
      }
    };
    
    Blockly.PHP['wm_reset'] = function(block) {
      var code = 'reset();\n';
      return code;
    };
    
    Blockly.Blocks['wm_session_destroy'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('destroy current session variables');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Destroy current session variables");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/session_destroy");
      }
    };
    
    Blockly.PHP['wm_session_destroy'] = function(block) {
      var code = 'session_destroy();\n';
      return code;
    };
    
    Blockly.Blocks['wm_session_start'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('start a new session');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("start a new session");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/session_start");
      }
    };
    
    Blockly.PHP['wm_session_start'] = function(block) {
      var code = 'session_start();\n';
      return code;
    };
    
     Blockly.Blocks['wm_session_is_new'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('is session new');
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setOutput(true, null);
        this.setTooltip("Check if session is new");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/session_is_new");
      }
    };
    
    Blockly.PHP['wm_session_is_new'] = function(block) {
      var code = 'session_is_new()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_process_kill'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("Number")
            .appendField('kill a process with id');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Kill a process");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/process_kill");
      }
    };
    
    Blockly.PHP['wm_process_kill'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'process_kill(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_nvram_backup'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('backup NVRAM to file');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Backup NVRAM to a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_backup");
      }
    };
    
    Blockly.PHP['wm_nvram_backup'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'nvram_backup(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_nvram_restore'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('restore NVRAM from file');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Restore NVRAM from a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_restore");
      }
    };
    
    Blockly.PHP['wm_nvram_restore'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'nvram_restore(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_nvram_defrag'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('defragment NVRAM');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Defragment NVRAM");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_defrag");
      }
    };
    
    Blockly.PHP['wm_nvram_defrag'] = function(block) {
      var code = 'nvram_defrag();\n';
      return code;
    };
    
    Blockly.Blocks['wm_nvram_dump'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('dump NVRAM to output');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Dump NVRAM to standard output");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_dump");
      }
    };
    
    Blockly.PHP['wm_nvram_dump'] = function(block) {
      var code = 'nvram_dump();\n';
      return code;
    };
    
    
    
    
    Blockly.Blocks['wm_nvram_get'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('get value from NVRAM for key');
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setOutput(true, null);
        this.setTooltip("Get a value from NVRAM for key");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_get");
      }
    };
    
    Blockly.PHP['wm_nvram_get'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'nvram_get(' + value_role_id  + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_nvram_unset'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('delete NVRAM value for key');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setInputsInline(true);
        this.setTooltip("Delete a key from NVRAM");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_unset");
      }
    };
    
    Blockly.PHP['wm_nvram_unset'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'nvram_set(' + value_role_id+');\n';
      return code;
    };
    
     Blockly.Blocks['wm_nvram_set'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck(null)
            .appendField('update NVRAM for key');
        this.appendValueInput("VAL")
            .setCheck(null)
            .appendField('with value ');    
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setInputsInline(true);
        this.setTooltip("Set a value from NVRAM for key");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_set");
      }
    };
    
    Blockly.PHP['wm_nvram_set'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var val = Blockly.PHP.valueToCode(block, 'VAL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'nvram_set(' + value_role_id  + ','+val+');\n';
      return code;
    };
    
    Blockly.Blocks['wm_nvram_free'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get NVRAM free space");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get NVRAM free space");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/nvram_free");
      }
    };
    
    Blockly.PHP['wm_nvram_free'] = function(block) {
      var code = 'nvram_free()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_mem_dump'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('dump memory to file for debugging');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Dump memory to file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mem_dump");
      }
    };
    
    Blockly.PHP['wm_mem_dump'] = function(block) {
      var code = 'mem_dump();\n';
      return code;
    };
    
    Blockly.Blocks['wm_mem_usage'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('dump memory usage to standard output');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Dump memory usage to standard output");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/mem_usage");
      }
    };
    
    Blockly.PHP['wm_mem_usage'] = function(block) {
      var code = 'mem_usage();\n';
      return code;
    };
    
    Blockly.Blocks['wm_net_disable3g'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('disable internet on dongle');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Disable cellular internet support on dongle");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/net_disable3g");
      }
    };
    
    Blockly.PHP['wm_net_disable3g'] = function(block) {
      var code = 'net_disable3g();\n';
      return code;
    };
    
    Blockly.Blocks['wm_wifi_disable'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('disable Wi-Fi module');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Disable Wi-Fi module");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/wifi_disable");
      }
    };
    
    Blockly.PHP['wm_wifi_disable'] = function(block) {
      var code = 'wifi_disable();\n';
      return code;
    };
    Blockly.Blocks['wm_wifi_enable'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('enable Wi-Fi module');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Enable Wi-Fi module");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/wifi_enable");
      }
    };
    
    Blockly.PHP['wm_wifi_enable'] = function(block) {
      var code = 'wifi_enable();\n';
      return code;
    };
     Blockly.Blocks['wm_net_enable3g'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('enable internet on dongle');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Enable cellular internet support on dongle");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/net_enable3g");
      }
    };
    
    Blockly.PHP['wm_net_enable3g'] = function(block) {
      var code = 'net_enable3g();\n';
      return code;
    };
    
    Blockly.Blocks['wm_max_execution_time'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("Number")
            .appendField('set max execution time');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set max script execution time in seconds");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/max_execution_time");
      }
    };
    
    Blockly.PHP['wm_max_execution_time'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'max_execution_time(' + value_role_id  + ');\n';
      return code;
    };
    
     Blockly.Blocks['wm_include'] = {
      init: function() {
        this.appendValueInput("STR")
            .setCheck("String")
            .appendField('include file');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Include a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/include");
      }
    };
    
    Blockly.PHP['wm_include'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'STR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'include(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_get3gstat'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get cellular status");
        this.setOutput(true, null);        
   //     this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get cellular status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/get3gstat");
      }
    };
    
    Blockly.PHP['wm_get3gstat'] = function(block) {
      var code = 'get3gstat()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_getmac'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get MAC address");
        this.setOutput(true, null);        
   //     this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get MAC address");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/getmac");
      }
    };
    
    Blockly.PHP['wm_getmac'] = function(block) {
      var code = 'getmac()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_getethstat'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get ethernet status");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get ethernet status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/getethstat");
      }
    };
    
    Blockly.PHP['wm_getethstat'] = function(block) {
      var code = 'getethstat()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_stats'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get system stats");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get system stats");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/stats");
      }
    };
    
    Blockly.PHP['wm_stats'] = function(block) {
      var code = 'stats()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_process_list'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get list of processes as array");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get process list");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/process_list");
      }
    };
    
    Blockly.PHP['wm_process_list'] = function(block) {
      var code = 'process_list()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };


    Blockly.Blocks['wm_netstat'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get network information as array");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get network status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/netstat");
      }
    };
    
    Blockly.PHP['wm_netstat'] = function(block) {
      var code = 'netstat()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_getusbstat'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get USB status");
        this.setOutput(true, null);        
    //    this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get USB status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/getusbstat");
      }
    };
    
    Blockly.PHP['wm_getusbtat'] = function(block) {
      var code = 'getusbstat()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    Blockly.Blocks['wm_phpinfo'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get information about the system");
        this.setOutput(true, null);        
    //    this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get information about the system - phpinfo()");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/phpinfo");
      }
    };
    
    Blockly.PHP['wm_phpinfo'] = function(block) {
      var code = 'phpinfo()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    
    Blockly.Blocks['wm_getwifistat'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("get Wi-Fi status");
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Get Wi-Fi status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/getwifistat");
      }
    };
    
    Blockly.PHP['wm_getwifistat'] = function(block) {
      var code = 'getwifistat()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_getcwd'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("current directory");
        this.setOutput(true, ["String"]);        
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Get current directory");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/getcwd");
      }
    };
    
    Blockly.PHP['wm_getcwd'] = function(block) {
      var code = 'getcwd()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ow_first'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("find first 1-Wire sensor")
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Find first 1-Wire sensor");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_first");
      }
    };
    
    Blockly.PHP['wm_ow_first'] = function(block) {
      var code = 'ow_first()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ow_next'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("find next 1-Wire sensor")
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Find next 1-Wire sensor");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_first");
      }
    };
    
    Blockly.PHP['wm_ow_next'] = function(block) {
      var code = 'ow_next()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ow_read'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("read byte from 1-Wire bus")
        this.setOutput(true, null);        
        this.setColour(65);
        this.setTooltip("Read byte from 1-Wire bus");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_read");
      }
    };
    
    Blockly.PHP['wm_ow_read'] = function(block) {
      var code = 'ow_read()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ow_read_temp'] = {
      init: function() {
        this.appendValueInput("VAL")
            .setCheck("Number")
            .appendField('read temperature from 1-Wire address');        
            
        this.setOutput(true, "Number");        
        this.setColour(65);
        this.setTooltip("Read temperature from 1-Wire bus");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_read");
      }
    };
    
    Blockly.PHP['wm_ow_read_temp'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ow_read_temp(' + value_role_id + ');\n';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ow_reset'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("reset 1-Wire bus")
        //this.setOutput(true, null);        
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(65);
        this.setTooltip("Reset 1-Wire bus");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_reset");
      }
    };
    
    Blockly.PHP['wm_ow_reset'] = function(block) {
      var code = 'ow_reset();\n';
      return code;
    };
    
    Blockly.Blocks['wm_ow_write'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("write value ")
        this.appendValueInput("VAL")
            .setCheck("Number")
            .appendField('to 1-Wire bus');        
            
        this.setInputsInline(true);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(65);
        this.setTooltip("Write byte to  1-Wire bus");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ow_write");
      }
    };
    
    Blockly.PHP['wm_ow_write'] = function(block) {
     var value_role_id = Blockly.PHP.valueToCode(block, 'VAL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ow_write(' + value_role_id + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_chdir'] = {
      init: function() {
        this.appendValueInput("DIR")
            .setCheck("String")
            .appendField('change directory to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Change directory");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/chdir");
      }
    };
    Blockly.PHP['wm_chdir'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'DIR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'chdir(' + value_role_id + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_chr'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('get character from ASCII code ');
        this.setOutput(true, ["String"]);
        this.setColour(64);
        this.setTooltip("Return the character for an ASCII code");
      }
    };
    Blockly.PHP['wm_chr'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'chr('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    Blockly.Blocks['wm_ord'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('get ASCII character from code');
        this.setOutput(true, ["String"]);
        this.setColour(64);
        this.setTooltip("Get ASCII character from code");
      }
    };
    Blockly.PHP['wm_ord'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ord('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    Blockly.Blocks['wm_die'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("die");
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("End current script");
      }
    };
    Blockly.PHP['wm_die'] = function(block) {
      var code = 'die()\r\n';
      return code;
    };
    

    Blockly.Blocks['wm_ping'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('ping host or ip ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Ping a host or IP");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/ping");
      }
    };
    Blockly.PHP['wm_ping'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ping('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    
    Blockly.Blocks['wm_disk_status'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("Number")
            .appendField('get mount status');
        this.setOutput(true, ["Number", "Boolean"]);
        this.setColour(64);
        this.setTooltip("Get disk mount status");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/disk_status");
      }
    };
    Blockly.PHP['wm_disk_status'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'disk_status('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    Blockly.Blocks['wm_disk_free_space'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('disk free space');
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Return the free space on the SD card");
      }
    };
    Blockly.PHP['wm_disk_free_space'] = function(block) {
      var code = 'disk_free_space()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_disk_total_space'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('disk total space');
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Return the total space on the SD card");
      }
    };
    Blockly.PHP['wm_disk_total_space'] = function(block) {
      var code = 'disk_total_space()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_power'] = {
      init: function() {
        this.appendValueInput("BASE")
            .setCheck("Number")
            .appendField('power with base of ');
        this.appendValueInput("EXP")
            .setCheck(null)
            .appendField('and exponent');
        this.setOutput(true,"Number");
        this.setInputsInline(true);
        this.setColour('%{BKY_MATH_HUE}');
        this.setTooltip("Calculate the power of a number");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/power");
      }
    };
    Blockly.PHP['wm_power'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'BASE', Blockly.PHP.ORDER_ATOMIC);
      var value_role_val = Blockly.PHP.valueToCode(block, 'EXP', Blockly.PHP.ORDER_ATOMIC);
      var code = 'power(' + value_role_id + ', ' + value_role_val + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_error_reporting'] = {
      init: function() {
        this.appendValueInput("VALUE")
            .setCheck(null)
            .appendField('set error reporting level');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Set the error reporting level");
      }
    };
    Blockly.PHP['wm_error_reporting'] = function(block) {
      var value_role_val = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      var code = 'error_reporting(' + value_role_val + ');\n';
      return code;
    };
    
    
    Blockly.Blocks['wm_exec'] = {
      init: function() {
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('execute script');
        this.appendValueInput("INTERVAL")
            .setCheck(null)
            .appendField('after delay of');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Execute a script");
      }
    };
    Blockly.PHP['wm_exec'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var value_role_val = Blockly.PHP.valueToCode(block, 'INTERVAL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'exec(' + value_role_id + ', ' + value_role_val + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_exec_action'] = {
      init: function() {
        this.appendValueInput("ACT")
            .setCheck(null)
            .appendField('trigger action');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Trigger an action");
      }
    };
    Blockly.PHP['wm_exec_action'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'ACT', Blockly.PHP.ORDER_ATOMIC);
      var code = 'exec_action(' + value_role_id + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_explode'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('split string');
        this.appendValueInput("DEL")
            .setCheck("String")
            .appendField('into array at character ');
        this.setInputsInline(true);    
        this.setOutput(true, ["Array"]);
        this.setColour(260);
        this.setTooltip("Return array of elements split at specified character");
      }
    };
    Blockly.PHP['wm_explode'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var idx_id = Blockly.PHP.valueToCode(block, 'DEL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'explode('+value_role_id+','+idx_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_tar_finish'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("Number")
            .appendField('finish TAR process with file handle ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("finish TAR process");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/tar_finish");
      }
    };
    Blockly.PHP['wm_tar_finish'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'tar_finish('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_tar_put'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("Number")
            .appendField('add to TAR file with handle');
        this.appendValueInput("SRC")
            .setCheck("String")
            .appendField('file and path');
        this.appendValueInput("DST")
            .setCheck("String")
            .appendField('path in TAR');
            
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("add a file to the TAR file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/tar_put");
      }
    };
    Blockly.PHP['wm_tar_put'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var src = Blockly.PHP.valueToCode(block, 'SRC', Blockly.PHP.ORDER_ATOMIC);
      var dst = Blockly.PHP.valueToCode(block, 'DST', Blockly.PHP.ORDER_ATOMIC);
      var code = 'tar_put('+value_role_id+','+src+','+dst+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fclose'] = {
      init: function() {
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('close ');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Close a file, socket or stream");
      }
    };
    Blockly.PHP['wm_fclose'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fclose(' + value_role_id + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_feof'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('is end of file ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Check if there is more data to read or not");
      }
    };
    Blockly.PHP['wm_feof'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'feof('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fgets'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('read line from');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get a line of data from the input file/socket/stream");
      }
    };
    Blockly.PHP['wm_fgets'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fgets('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_file_exists'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('file exists ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Check if the specified file exists");
      }
    };
    Blockly.PHP['wm_file_exists'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'file_exists('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_filesize'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('file size ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get size of file/stream/socket");
      }
    };
    Blockly.PHP['wm_filesize'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'filesize('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_findfirst'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('find first file matching ');
        this.appendValueInput("FLAGS")
            .setCheck("Number")
            .appendField('with flag ');
            
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Find the first instance of a file pattern matching flags");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/findfirst");
      }
    };
    Blockly.PHP['wm_findfirst'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'findfirst('+value_role_id+','+flags+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_findnext'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('find next instance of file');
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Find the next instance of a file ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/findnext");
      }
    };
    Blockly.PHP['wm_findnext'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'findnext()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_strval'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('convert value ');
        this.appendDummyInput()
            .appendField('to string');    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get value as a string");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/floatval");
      }
    };
    Blockly.PHP['wm_strval'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'strval('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_floatval'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('convert value to float ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get value as a float");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/floatval");
      }
    };
    Blockly.PHP['wm_floatval'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'floatval('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_intval'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('convert value to int ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get value as an int");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/intval");
      }
    };
    Blockly.PHP['wm_intval'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'intval('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_flush'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("flush output");
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(64);
        this.setTooltip("Flush output to browser ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/flush");
      }
    };
    Blockly.PHP['wm_flush'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'flush();\r\n';
      return code;
    };
    
    Blockly.Blocks['wm_firmwareupdate'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("initiate firmware update");
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(64);
        this.setTooltip("Update firmware ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/firmwareupdate");
      }
    };
    Blockly.PHP['wm_firmwareupdate'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'firmwareupdate();\r\n';
      return code;
    };
    
    Blockly.Blocks['wm_debugout'] = {
      init: function() {
         this.appendValueInput("VAR")
            .setCheck(["Number"])
            .appendField('set debug output to ');
        this.setOutput(true, null);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(64);
        this.setTooltip("Set debug output state ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/debugout");
      }
    };
    Blockly.PHP['wm_debugout'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'debugout('+value_role_id+');\r\n';
      return code;
    };
    
     Blockly.Blocks['wm_debug'] = {
      init: function() {
        this.appendValueInput("VAL")
            .setCheck(null)
            .appendField('print a debug message');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Print a debug message");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/debug");
      }
    };
    
    Blockly.PHP['wm_debug'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'debug(' + value_role_id  + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_clear_watchdog'] = {
      init: function() {
        this.appendDummyInput()
        .appendField("clear watchdog timer");
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(64);
        this.setTooltip("Clear watchdog timer ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/clear_watchdog");
      }
    };
    Blockly.PHP['wm_clear_watchdog'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'clear_watchdog();\r\n';
      return code;
    };
    
    Blockly.Blocks['wm_init_watchdog'] = {
      init: function() {
       this.appendValueInput("VAL")
            .setCheck("Number")
            .appendField("set watchdog timer interval to");
         this.appendDummyInput()
        .appendField("seconds");    
        this.setInputsInline(true);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        
        this.setColour(64);
        this.setTooltip("Init watchdog timer ");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/init_watchdog");
      }
    };
    Blockly.PHP['wm_init_watchdog'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'init_watchdog('+value_role_id+');\r\n';
      return code;
    };
    
    Blockly.Blocks['wm_f485open'] = {
      init: function() {
        this.appendValueInput("BAUD")
            .setCheck("Number")
            .appendField('open RS-485 port with baud rate');
        this.appendDummyInput().appendField('and')
            .appendField(new Blockly.FieldDropdown([["even","0"], ["odd","1"]]), "PARITY").appendField('parity');
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Open RS-485 Port");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/f485open");
      }
    };
    Blockly.PHP['wm_f485open'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'BAUD', Blockly.PHP.ORDER_ATOMIC);
      var value_parity = block.getFieldValue('PARITY');
      var code = 'f485open('+value_role_id+','+value_parity+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fopen'] = {
      init: function() {
        this.appendValueInput("FN")
            .setCheck("String")
            .appendField('open a file');
        this.appendDummyInput().appendField('as')
            .appendField(new Blockly.FieldDropdown([["read-only","r"], ["write/overwrite","w"], ["read/write","a"]]), "STAT");
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Open a file or stream");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fopen");
      }
    };
    Blockly.PHP['wm_fopen'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
//      var value_perm = Blockly.PHP.valueToCode(block, 'STAT', Blockly.PHP.ORDER_ATOMIC);
      var value_3 = block.getFieldValue('STAT');
      var code = 'fopen('+value_role_id+',"'+value_3+'")';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fread'] = {
      init: function() {
        this.appendValueInput("CNT")
            .setCheck("Number")
            .appendField('read ');
          
        this.appendValueInput("FN")
            .setCheck("Number")
            .appendField('bytes from file ');
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Read from a file or stream");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fread");
      }
    };
    Blockly.PHP['wm_fread'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var value2 = Blockly.PHP.valueToCode(block, 'CNT', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fread('+value_role_id+','+value2+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_freemem'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('free memory');
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Find the remaining memory free");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/freemem");
      }
    };
    Blockly.PHP['wm_freemem'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'freemem()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_freestack'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('free stack');
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Find the remaining stack space");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/freestack");
      }
    };
    Blockly.PHP['wm_freemem'] = function(block) {
      //var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      //var flags = Blockly.PHP.valueToCode(block, 'FLAGS', Blockly.PHP.ORDER_ATOMIC);
      var code = 'freemem()';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_rename'] = {
      init: function() {
        this.appendValueInput("SRC")
            .setCheck("String")
            .appendField('rename file');
        this.appendValueInput("DEST")
            .setCheck("String")
            .appendField('to');


        this.setInputsInline(true);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Rename a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/rename");
      }
    };
    
    Blockly.PHP['wm_rename'] = function(block) {
      var value_1 = Blockly.PHP.valueToCode(block, 'SRC', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = Blockly.PHP.valueToCode(block, 'DEST', Blockly.PHP.ORDER_ATOMIC);
      var code = 'rename(' + value_1 + ', ' + value_2 + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_rmdir'] = {
      init: function() {
        this.appendValueInput("PATH")
            .setCheck("String")
            .appendField('remove directory ');
        this.appendDummyInput().appendField('and')
            .appendField(new Blockly.FieldDropdown([["do not delete subdirectories","0"], ["delete subdirectories","1"]]), "DEL");
        this.setOutput(true,"Number");
        this.setInputsInline(true);
        //this.setPreviousStatement(true, null);
        //this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Remove a directory");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/rmdir");
      }
    };
    
    Blockly.PHP['wm_rmdir'] = function(block) {
      var value_1 = Blockly.PHP.valueToCode(block, 'PATH', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = block.getFieldValue('DEL');
      var code = 'rmdir(' + value_1 + ', ' + value_2 + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_unlink'] = {
      init: function() {
        this.appendValueInput("PATH")
            .setCheck("String")
            .appendField('delete file ');
        this.setOutput(true,"Number");
        this.setInputsInline(true);
        this.setColour(65);
        this.setTooltip("Delte file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/unlink");
      }
    };
    
    Blockly.PHP['wm_unlink'] = function(block) {
      var value_1 = Blockly.PHP.valueToCode(block, 'PATH', Blockly.PHP.ORDER_ATOMIC);
      var code = 'unlink(' + value_1 + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_send_sms'] = {
      init: function() {
        this.appendValueInput("NUM")
            .setCheck("String")
            .appendField('send SMS to number ');
        this.appendValueInput("MSG")
            .setCheck("String")
            .appendField('with message');
        this.setOutput(true,"Number");
        this.setInputsInline(true);
        this.setColour(65);
        this.setTooltip("Send an SMS");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/send_sms");
      }
    };
    
    Blockly.PHP['wm_send_sms'] = function(block) {
      var value_1 = Blockly.PHP.valueToCode(block, 'NUM', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = Blockly.PHP.valueToCode(block, 'MSG', Blockly.PHP.ORDER_ATOMIC);
      var code = 'send_sms(' + value_1 + ', ' + value_2 + ')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fseek'] = {
      init: function() {
        this.appendValueInput("FN")
            .setCheck("Number")
            .appendField('in file');
        this.appendValueInput("VALUE")
            .setCheck("Number")
            .appendField('seek to position');
        this.appendDummyInput().appendField('from')
            .appendField(new Blockly.FieldDropdown([["beginning of file","0"], ["current position","1"], ["end of file","2"]]), "FROM");
            

        this.setInputsInline(true);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Seek to a position in the file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fseek");
      }
    };
    
    Blockly.PHP['wm_fseek'] = function(block) {
      var value_1 = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      //var value_3 = Blockly.PHP.valueToCode(block, 'FROM', Blockly.PHP.ORDER_ATOMIC);
      var value_3 = block.getFieldValue('FROM');
      var code = 'fseek(' + value_1 + ', ' + value_2 + ', ' +value_3 + ');\n';
      return code;
    };
    
    Blockly.Blocks['wm_fseropen'] = {
      init: function() {
        this.appendValueInput("BAUD")
            .setCheck("Number")
            .appendField('open serial port with baud rate');
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Open Serial Port");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fseropen");
      }
    };
    Blockly.PHP['wm_fseropen'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'BAUD', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fseropen('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_fsockopen'] = {
      init: function() {
        this.appendValueInput("HOST")
            .setCheck("String")
            .appendField('open a socket to host ');
        this.appendValueInput("PORT")
            .setCheck("Number")
            .appendField('and port');
            
        this.appendValueInput("TIMEOUT")
            .setCheck("Number")
            .appendField('with timeout of');
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Open a socket");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fsockopen");
      }
    };
    Blockly.PHP['wm_fsockopen'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'HOST', Blockly.PHP.ORDER_ATOMIC);
      var value_port = Blockly.PHP.valueToCode(block, 'PORT', Blockly.PHP.ORDER_ATOMIC);
      var value_perm = Blockly.PHP.valueToCode(block, 'TIMEOUT', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fsockopen('+value_role_id+','+value_port+','+value_perm+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_ftell'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('get position in file ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Get current file position");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/ftell");
      }
    };
    Blockly.PHP['wm_ftell'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ftell('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    Blockly.Blocks['wm_fwrite'] = {
      init: function() {
        this.appendValueInput("DAT")
            .setCheck(null)
            .appendField('write data ');
        this.appendValueInput("FN")
            .setCheck(null)
            .appendField('to file');
            
        this.setInputsInline(true);
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(64);
        this.setTooltip("Write to a file");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fwrite");
      }
    };
    Blockly.PHP['wm_fwrite'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = Blockly.PHP.valueToCode(block, 'DAT', Blockly.PHP.ORDER_ATOMIC);
      var inp=block.getInput();
      var code = 'fwrite('+value_role_id+','+value_2+')';
      if (inp)
        return [code, Blockly.PHP.ORDER_ATOMIC];
      else return code+";\r\n";
    };
     Blockly.Blocks['wm_fwrite2'] = {
      init: function() {
        this.appendValueInput("DAT")
            .setCheck(null)
            .appendField('write data ');
        this.appendValueInput("FN")
            .setCheck(null)
            .appendField('to file');
        this.appendDummyInput().appendField('and return result code');    
        this.setInputsInline(true);
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Write to a file and return status");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/fwrite");
      }
    };
    Blockly.PHP['wm_fwrite2'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'FN', Blockly.PHP.ORDER_ATOMIC);
      var value_2 = Blockly.PHP.valueToCode(block, 'DAT', Blockly.PHP.ORDER_ATOMIC);
      var code = 'fwrite('+value_role_id+','+value_2+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };


    Blockly.Blocks['wm_implode'] = {
      init: function() {
        this.appendValueInput("ARR")
            .setCheck("Array")
            .appendField('Turn array');
        this.appendValueInput("DEL")
            .setCheck("String")
            .appendField('into string using delimeter');    
        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(260);
        this.setTooltip("Turn an array into a string");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/implode");
      }
    };
    Blockly.PHP['wm_implode'] = function(block) {
      var v1 = Blockly.PHP.valueToCode(block, 'ARR', Blockly.PHP.ORDER_ATOMIC);
      var v2 = Blockly.PHP.valueToCode(block, 'DEL', Blockly.PHP.ORDER_ATOMIC);
      var code = 'implode('+v1+','+v2+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_json_encode'] = {
      init: function() {
        this.appendValueInput("ARR")
            .setCheck("Array")
            .appendField('JSON-encode array');
         this.appendDummyInput()
            .appendField('as ')
            .appendField(new Blockly.FieldDropdown([["object","0"], ["array","1"]]), "METHOD");

        this.setInputsInline(true);    
        this.setOutput(true, null);
        this.setColour(260);
        this.setTooltip("Return JSON-encoded string");
        this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/json_encode");
      }
    };
    Blockly.PHP['wm_json_encode'] = function(block) {
      var v1 = Blockly.PHP.valueToCode(block, 'ARR', Blockly.PHP.ORDER_ATOMIC);
      var value_role_id =block.getFieldValue('METHOD');
      var code = 'json_encode('+v1+','+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    Blockly.Blocks['wm_function_exists'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck("String")
            .appendField('if function exists ');
        this.setOutput(true, null);
        this.setColour(64);
        this.setTooltip("Check if a function name exists");
         this.setHelpUrl("http://www.wattmon.com/dokuwiki/uphp/functions/function_exists");
      }
    };
    Blockly.PHP['wm_function_exists'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'function_exists('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };

    //============
    Blockly.Blocks['wm_ieee754toint'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('convert float');
            
        this.appendValueInput("VAR")
            .setCheck("Number");
        this.appendDummyInput()    
            .appendField(' to IEEE754 encoded int ');
            
        this.setInputsInline(true);
        this.setOutput(true, ["Number"]);
        this.setColour(64);
        this.setTooltip("Convert a float value to an IEEE-754 encoded integer (32 bit)");
      }
    };
    Blockly.PHP['wm_ieee754toint'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'ieee754toint('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_inttoieee754'] = {
      init: function() {
        this.appendDummyInput()
            .appendField('convert int');
            
        this.appendValueInput("VAR")
            .setCheck("Number");
        this.appendDummyInput()    
            .appendField(' to IEEE754 float ');
            
        this.setInputsInline(true);
        this.setOutput(true, ["Number"]);
        this.setColour(64);
        this.setTooltip("Convert an IEEE-754 encoded integer representation (32 bit) to a float");
      }
    };
    Blockly.PHP['wm_inttoieee754'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'inttoieee754('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_htmlspecialchars'] = {
      init: function() {
        this.appendValueInput("VAR")
            .setCheck(null)
            .appendField('replace special characters for HTML ');
        this.setOutput(true, ["String"]);
        this.setColour(64);
        this.setTooltip("Convert special characters for display in HTML");
      }
    };
    Blockly.PHP['wm_htmlspecialchars'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'VAR', Blockly.PHP.ORDER_ATOMIC);
      var code = 'htmlspecialchars('+value_role_id+')';
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    
    Blockly.Blocks['wm_var_global'] = {
      init: function() {
        this.appendValueInput("VALUE")
            .setCheck("Variable")
            .appendField('use global variable ');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(65);
        this.setTooltip("Use script global context variable");
      }
    };
    
    Blockly.PHP['wm_var_global'] = function(block) {
      var value_role_val = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      var code = 'global '+ value_role_val + ';\n';
      return code;
    };
    
    // get global variable
    Blockly.Blocks['wm_global_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('GLOBAL variables');
    this.setOutput(true, null);            
        this.setColour(330);
     this.setTooltip("Get the value of a global variable");
     
      }
    };
    Blockly.PHP['wm_global_variable'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code='';
      if (value_role_id) {
        code = '$_GLOBALS['+value_role_id+']';
      } else code = '$_GLOBALS';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    // Set global variable
    Blockly.Blocks['wm_set_global_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('set GLOBAL variable');
        this.appendValueInput("VALUE")
            .setCheck(null)
            .appendField('to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(330);
     this.setTooltip("Set the value of a global variable");
     
      }
    };
    Blockly.PHP['wm_set_global_variable'] = function(block) {
      var name_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      var value_role_id = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = '$_GLOBALS['+name_role_id+']='+value_role_id+';\n';
      // TODO: Change ORDER_NONE to the correct strength.
      return code;
    };
    //Blockly.Variables.createVariable(Code.workspace, null, 'panda');

// $get  variable
    Blockly.Blocks['wm_get_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('GET variables ');
        this.setOutput(true, null);            
        this.setColour(330);
     this.setTooltip("Get the value of a $_GET variable");
     
      }
    };
    Blockly.PHP['wm_get_variable'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code='';
      if (value_role_id) {
        code = '$_GET['+value_role_id+']';
      } else code = '$_GET';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
// $get  variable
    Blockly.Blocks['wm_post_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('POST variables ');
    this.setOutput(true, null);            
        this.setColour(330);
     this.setTooltip("Get the value of a $_POST variable");
     
      }
    };
    Blockly.PHP['wm_post_variable'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code='';
      if (value_role_id) {
        code = '$_POST['+value_role_id+']';
      } else code = '$_POST';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };    
    // $get  variable
    Blockly.Blocks['wm_server_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck("String")
            .appendField('SERVER variables ');
        this.setOutput(true, null);            
        this.setColour(330);
     this.setTooltip("Get the value of a $_SERVER variable");
     
      }
    };
    Blockly.PHP['wm_server_variable'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code='';
      if (value_role_id) {
        code = '$_SERVER['+value_role_id+']';
      } else code = '$_SERVER';
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    // Set global variable
    Blockly.Blocks['wm_set_array_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck(null)
            .appendField('set item in list');
        this.appendValueInput("INDEX")
            .setCheck(null)
            .appendField('at index');
        this.appendValueInput("VALUE")
            .setCheck(null)
            .appendField('to');
        this.setPreviousStatement(true, null);
        this.setNextStatement(true, null);
        this.setColour(260);
        this.setInputsInline(true);
        this.setTooltip("Set an item in a list");
     
      }
    };
    Blockly.PHP['wm_set_array_variable'] = function(block) {
      var name_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      var index_id = Blockly.PHP.valueToCode(block, 'INDEX', Blockly.PHP.ORDER_ATOMIC);
      var value_role_id = Blockly.PHP.valueToCode(block, 'VALUE', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      var code = '$'+name_role_id+'[' + index_id + ']='+value_role_id+';\n';
      if (name_role_id.substr(0,1)=='$') {  // special case for $_GLOBALS
          code=code.substr(1);
      }
      // TODO: Change ORDER_NONE to the correct strength.
      return code;
    };
    //Blockly.Variables.createVariable(Code.workspace, null, 'panda');

// $get  variable
    Blockly.Blocks['wm_array_variable'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck(null)
            .appendField('get value from array ')
        this.appendValueInput("INDEX")
            .setCheck(null)
            .setAlign(Blockly.ALIGN_RIGHT)
            .appendField('at key');
         this.setInputsInline(true);

        this.setOutput(true, null);            
        this.setColour(260);
     this.setTooltip("Get an item from a list");
     
      }
    };
   Blockly.PHP['wm_array_variable'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      var index_id = Blockly.PHP.valueToCode(block, 'INDEX', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      
      var code = '$'+value_role_id+'['+index_id+']';
      if (value_role_id.substr(0,1)=='$') {  // special case for $_GLOBALS
          code=code.substr(1);
      }
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
     Blockly.Blocks['wm_array_key'] = {
      init: function() {
        this.appendValueInput("NAME")
            .setCheck(null)
            .appendField('get key of array ')
        this.appendValueInput("INDEX")
            .setCheck(null)
            .setAlign(Blockly.ALIGN_RIGHT)
            .appendField('at index');
         this.setInputsInline(true);

        this.setOutput(true, null);            
        this.setColour(260);
        this.setTooltip("Return the key for an array index");
        this.setHelpUrl("http://wattmon.com/dokuwiki/uphp/functions/array_key");
     
      }
    };
    
    
   Blockly.PHP['wm_array_key'] = function(block) {
      var value_role_id = Blockly.PHP.valueToCode(block, 'NAME', Blockly.PHP.ORDER_ATOMIC);
      var index_id = Blockly.PHP.valueToCode(block, 'INDEX', Blockly.PHP.ORDER_ATOMIC);
      // TODO: Assemble PHP into code variable.
      if (value_role_id.substr(0,1)=='$') {  // special case for $_GLOBALS
          value_role_id=value_role_id.substr(1);
      }
      
      var code = 'array_key($'+value_role_id+','+index_id+')';
      
      // TODO: Change ORDER_NONE to the correct strength.
      return [code, Blockly.PHP.ORDER_ATOMIC];
    };
    
    // override PHP functions specific to Wattmon - this is required because Wattmon's implode is different
    
    Blockly.PHP['text_join'] = function(block) {
      // Create a string made up of any number of elements of any type.
      if (block.itemCount_ == 0) {
        return ['\'\'', Blockly.PHP.ORDER_ATOMIC];
      } else if (block.itemCount_ == 1) {
        var element = Blockly.PHP.valueToCode(block, 'ADD0',
            Blockly.PHP.ORDER_NONE) || '\'\'';
        var code = element;
        return [code, Blockly.PHP.ORDER_FUNCTION_CALL];
      } else if (block.itemCount_ == 2) {
        var element0 = Blockly.PHP.valueToCode(block, 'ADD0',
            Blockly.PHP.ORDER_NONE) || '\'\'';
        var element1 = Blockly.PHP.valueToCode(block, 'ADD1',
            Blockly.PHP.ORDER_NONE) || '\'\'';
        var code = element0 + ' . ' + element1;
        return [code, Blockly.PHP.ORDER_ADDITION];
      } else {
        var elements = new Array(block.itemCount_);
        for (var i = 0; i < block.itemCount_; i++) {
          elements[i] = Blockly.PHP.valueToCode(block, 'ADD' + i,
              Blockly.PHP.ORDER_COMMA) || '\'\'';
        }
        var code = 'implode(array(' + elements.join(',') + '),\'\')';
        return [code, Blockly.PHP.ORDER_FUNCTION_CALL];
      }
    };
    
    // override function generation - variables are all local
    Blockly.PHP['procedures_defreturn'] = function(block) {
      // Define a procedure with a return value.
      // First, add a 'global' statement for every variable that is not shadowed by
      // a local parameter.
      var globals = [];
      var varName;
      var workspace = block.workspace;
      var variables = Blockly.Variables.allUsedVarModels(workspace) || [];
      for (var i = 0, variable; variable = variables[i]; i++) {
        varName = variable.name;
        if (block.arguments_.indexOf(varName) == -1) {
          globals.push(Blockly.PHP.variableDB_.getName(varName,
              Blockly.Variables.NAME_TYPE));
        }
      }
      // Add developer variables.
      var devVarList = Blockly.Variables.allDeveloperVariables(workspace);
      for (var i = 0; i < devVarList.length; i++) {
        globals.push(Blockly.PHP.variableDB_.getName(devVarList[i],
            Blockly.Names.DEVELOPER_VARIABLE_TYPE));
      }
      //globals = globals.length ?
        //  Blockly.PHP.INDENT + 'global ' + globals.join(', ') + ';\n' : '';
    
      var funcName = Blockly.PHP.variableDB_.getName(
          block.getFieldValue('NAME'), Blockly.Procedures.NAME_TYPE);
      var branch = Blockly.PHP.statementToCode(block, 'STACK');
      if (Blockly.PHP.STATEMENT_PREFIX) {
        var id = block.id.replace(/\$/g, '$$$$');  // Issue 251.
        branch = Blockly.PHP.prefixLines(
            Blockly.PHP.STATEMENT_PREFIX.replace(
                /%1/g, '\'' + id + '\''), Blockly.PHP.INDENT) + branch;
      }
      if (Blockly.PHP.INFINITE_LOOP_TRAP) {
        branch = Blockly.PHP.INFINITE_LOOP_TRAP.replace(/%1/g,
            '\'' + block.id + '\'') + branch;
      }
      var returnValue = Blockly.PHP.valueToCode(block, 'RETURN',
          Blockly.PHP.ORDER_NONE) || '';
      if (returnValue) {
        returnValue = Blockly.PHP.INDENT + 'return ' + returnValue + ';\n';
      }
      var args = [];
      for (var i = 0; i < block.arguments_.length; i++) {
        args[i] = Blockly.PHP.variableDB_.getName(block.arguments_[i],
            Blockly.Variables.NAME_TYPE);
      }
      var code = 'function ' + funcName + '(' + args.join(', ') + ') {\n' +
           branch + returnValue + '}'; //globals
      code = Blockly.PHP.scrub_(block, code);
      // Add % so as not to collide with helper functions in definitions list.
      Blockly.PHP.definitions_['%' + funcName] = code;
      return null;
    };
    
    // Defining a procedure without a return value uses the same generator as
    // a procedure with a return value.
    Blockly.PHP['procedures_defnoreturn'] =
        Blockly.PHP['procedures_defreturn'];
// $get  variable    


    Blockly.Variables.flyoutCategoryBlocks = function(workspace) {
      var variableModelList = workspace.getVariablesOfType('');
      variableModelList.sort(Blockly.VariableModel.compareByName);
    
      var xmlList = [];
      if (variableModelList.length > 0) {
        var firstVariable = variableModelList[0];
        if (Blockly.Blocks['variables_set']) {
          var gap = Blockly.Blocks['math_change'] ? 8 : 24;
          var blockText = '<xml>' +
                '<block type="variables_set" gap="' + gap + '">' +
                Blockly.Variables.generateVariableFieldXmlString(firstVariable) +
                '</block>' +
                '</xml>';
          var block = Blockly.Xml.textToDom(blockText).firstChild;
          xmlList.push(block);
        }
        if (Blockly.Blocks['math_change']) {
          var gap = Blockly.Blocks['variables_get'] ? 20 : 8;
          var blockText = '<xml>' +
              '<block type="math_change" gap="' + gap + '">' +
              Blockly.Variables.generateVariableFieldXmlString(firstVariable) +
              '<value name="DELTA">' +
              '<shadow type="math_number">' +
              '<field name="NUM">1</field>' +
              '</shadow>' +
              '</value>' +
              '</block>' +
              '</xml>';
          var block = Blockly.Xml.textToDom(blockText).firstChild;
          xmlList.push(block);
        }
        var gap = Blockly.Blocks['variables_get'] ? 20 : 8;
        // added an option for a global keyword when defining the scope
        var blockText = '<xml>' +
              '<block type="wm_var_global" gap="' + gap + '">' +
              '</block>' +
              '</xml>';
         var block = Blockly.Xml.textToDom(blockText).firstChild;
         xmlList.push(block);
         
        for (var i = 0, variable; variable = variableModelList[i]; i++) {
          if (Blockly.Blocks['variables_get']) {
            var blockText = '<xml>' +
                '<block type="variables_get" gap="8">' +
                Blockly.Variables.generateVariableFieldXmlString(variable) +
                '</block>' +
                '</xml>';
            var block = Blockly.Xml.textToDom(blockText).firstChild;
            xmlList.push(block);
          }
        }
      }
      return xmlList;
    };
    
    Blockly.PHP['controls_for'] = function(block) {
      // For loop.
      var variable0 = Blockly.PHP.variableDB_.getName(
          block.getFieldValue('VAR'), Blockly.Variables.NAME_TYPE);
      var argument0 = Blockly.PHP.valueToCode(block, 'FROM',
          Blockly.PHP.ORDER_ASSIGNMENT) || '0';

      var argument1 = Blockly.PHP.valueToCode(block, 'TO',
          Blockly.PHP.ORDER_ASSIGNMENT) || '0';
      var increment = Blockly.PHP.valueToCode(block, 'BY',
          Blockly.PHP.ORDER_ASSIGNMENT) || '1';
      var branch = Blockly.PHP.statementToCode(block, 'DO');
      branch = Blockly.PHP.addLoopTrap(branch, block.id);
      var code;
      if (Blockly.isNumber(argument0) && Blockly.isNumber(argument1) &&
          Blockly.isNumber(increment)) {
        // All arguments are simple numbers.
        var up = parseFloat(argument0) <= parseFloat(argument1);
        code = 'for (' + variable0 + ' = ' + argument0 + '; ' +
            variable0 + (up ? ' <= ' : ' >= ') + argument1 + '; ' +
            variable0;
        var step = Math.abs(parseFloat(increment));
        if (step == 1) {
          code += up ? '++' : '--';
        } else {
          code += (up ? ' += ' : ' -= ') + step;
        }
        code += ') {\n' + branch + '}\n';
      } else {
        code = '';
        // Cache non-trivial values to variables to prevent repeated look-ups.
        var startVar = argument0;
        if (!argument0.match(/^\w+$/) && !Blockly.isNumber(argument0)) {
          startVar = variable0 + '_start';
          code += startVar + ' = ' + argument0 + ';\n';
        }
        var endVar = argument1;
        if (!argument1.match(/^\w+$/) && !Blockly.isNumber(argument1)) {
          var endVar=variable0 + '_end';
          code += endVar + ' = ' + argument1 + ';\n';
        }
        // Determine loop direction at start, in case one of the bounds
        // changes during loop execution.
        var incVar=variable0 + '_inc';
        
        // only generate a different name if this var alrady exists
        //alert(variable0 + 'inc'+"="+incVar);
        code += incVar + ' = ';
        if (Blockly.isNumber(increment)) {
          code += Math.abs(increment) + ';\n';
        } else {
          code += 'abs(' + increment + ');\n';
        }
        code += 'if (' + startVar + ' > ' + endVar + ') {\n';
        code += Blockly.PHP.INDENT + incVar + ' = -' + incVar + ';\n';
        code += '}\n';
        code += 'for (' + variable0 + ' = ' + startVar + '; ' +
            incVar + ' >= 0 ? ' +
            variable0 + ' <= ' + endVar + ' : ' +
            variable0 + ' >= ' + endVar + '; ' +
            variable0 + ' += ' + incVar + ') {\n' +
            branch + '}\n';
      }
      return code;
    };
    
    // change default count to sizeof
    Blockly.PHP['text_length'] = function(block) {
  // String or array length.
  var functionName = Blockly.PHP.provideFunction_(
      'length',
      ['function ' + Blockly.PHP.FUNCTION_NAME_PLACEHOLDER_ + '($value) {',
       '  if (is_string($value)) {',
       '    return strlen($value);',
       '  } else {',
       '    return sizeof($value);',
       '  }',
       '}']);
  var text = Blockly.PHP.valueToCode(block, 'VALUE',
      Blockly.PHP.ORDER_NONE) || '\'\'';
  return [functionName + '(' + text + ')', Blockly.PHP.ORDER_FUNCTION_CALL];
};
}

/**
 * Save the blocks and reload with a different language.
 */
Code.changeLanguage = function() {
  // Store the blocks for the duration of the reload.
  // MSIE 11 does not support sessionStorage on file:// URLs.
  if (window.sessionStorage) {
    var xml = Blockly.Xml.workspaceToDom(Code.workspace);
    var text = Blockly.Xml.domToText(xml);
    window.sessionStorage.loadOnceBlocks = text;
  }

  var languageMenu = document.getElementById('languageMenu');
  var newLang = encodeURIComponent(
      languageMenu.options[languageMenu.selectedIndex].value);
  var search = window.location.search;
  if (search.length <= 1) {
    search = '?lang=' + newLang;
  } else if (search.match(/[?&]lang=[^&]*/)) {
    search = search.replace(/([?&]lang=)[^&]*/, '$1' + newLang);
  } else {
    search = search.replace(/\?/, '?lang=' + newLang + '&');
  }

  window.location = window.location.protocol + '//' +
      window.location.host + window.location.pathname + search;
};

/**
 * Bind a function to a button's click event.
 * On touch enabled browsers, ontouchend is treated as equivalent to onclick.
 * @param {!Element|string} el Button element or ID thereof.
 * @param {!Function} func Event handler to bind.
 */
Code.bindClick = function(el, func) {
  if (typeof el == 'string') {
    el = document.getElementById(el);
  }
  el.addEventListener('click', func, true);
  el.addEventListener('touchend', func, true);
};

/**
 * Load the Prettify CSS and JavaScript.
 */
Code.importPrettify = function() {
  var script = document.createElement('script');
  script.setAttribute('src', 'https://cdn.rawgit.com/google/code-prettify/master/loader/run_prettify.js');
  document.head.appendChild(script);
};

/**
 * Compute the absolute coordinates and dimensions of an HTML element.
 * @param {!Element} element Element to match.
 * @return {!Object} Contains height, width, x, and y properties.
 * @private
 */
Code.getBBox_ = function(element) {
  var height = element.offsetHeight;
  var width = element.offsetWidth;
  var x = 0;
  var y = 0;
  do {
    x += element.offsetLeft;
    y += element.offsetTop;
    element = element.offsetParent;
  } while (element);
  return {
    height: height,
    width: width,
    x: x,
    y: y
  };
};

/**
 * User's language (e.g. "en").
 * @type {string}
 */
Code.LANG = Code.getLang();

/**
 * List of tab names.
 * @private
 */
Code.TABS_ = ['blocks', 'php', 'xml'];

Code.selected = 'blocks';

/**
 * Switch the visible pane when a tab is clicked.
 * @param {string} clickedName Name of tab clicked.
 */
Code.tabClick = function(clickedName) {
  // If the XML tab was open, save and render the content.
  if (document.getElementById('tab_xml').className == 'tabon') {
    var xmlTextarea = document.getElementById('content_xml');
    var xmlText = xmlTextarea.value;
    var xmlDom = null;
    try {
      xmlDom = Blockly.Xml.textToDom(xmlText);
    } catch (e) {
      var q =
          window.confirm(MSG['badXml'].replace('%1', e));
      if (!q) {
        // Leave the user on the XML tab.
        return;
      }
    }
    if (xmlDom) {
      Code.workspace.clear();
      Blockly.Xml.domToWorkspace(xmlDom, Code.workspace);
    }
  }

  if (document.getElementById('tab_blocks').className == 'tabon') {
    Code.workspace.setVisible(false);
  }
  // Deselect all tabs and hide all panes.
  for (var i = 0; i < Code.TABS_.length; i++) {
    var name = Code.TABS_[i];
    document.getElementById('tab_' + name).className = 'taboff';
    document.getElementById('content_' + name).style.visibility = 'hidden';
  }

  // Select the active tab.
  Code.selected = clickedName;
  document.getElementById('tab_' + clickedName).className = 'tabon';
  // Show the selected pane.
  document.getElementById('content_' + clickedName).style.visibility =
      'visible';
  Code.renderContent();
  if (clickedName == 'blocks') {
    Code.workspace.setVisible(true);
  }
  Blockly.svgResize(Code.workspace);
};

/**
 * Populate the currently selected pane with content generated from the blocks.
 */
Code.renderContent = function() {
  var content = document.getElementById('content_' + Code.selected);
  // Initialize the pane.
  if (content.id == 'content_xml') {
    var xmlTextarea = document.getElementById('content_xml');
    var xmlDom = Blockly.Xml.workspaceToDom(Code.workspace);
    var xmlText = Blockly.Xml.domToPrettyText(xmlDom);
    xmlTextarea.value = xmlText;
    xmlTextarea.focus();
  } else if (content.id == 'content_php') {
    Code.attemptCodeGeneration(Blockly.PHP, 'php');
  } 
};

/**
 * Attempt to generate the code and display it in the UI, pretty printed.
 * @param generator {!Blockly.Generator} The generator to use.
 * @param prettyPrintType {string} The file type key for the pretty printer.
 */
Code.attemptCodeGeneration = function(generator, prettyPrintType) {
  var content = document.getElementById('content_' + Code.selected);
  content.textContent = '';
  if (Code.checkAllGeneratorFunctionsDefined(generator)) {
    var code = generator.workspaceToCode(Code.workspace);
    code=code.replace(/\\\\/mg,"\\");
    code=code.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;') ;
    content.textContent = code;
    if (typeof PR.prettyPrintOne == 'function') {
      code = content.textContent;
      code = PR.prettyPrintOne(code, prettyPrintType);
      content.innerHTML = code;
    }
  }
};

/**
 * Check whether all blocks in use have generator functions.
 * @param generator {!Blockly.Generator} The generator to use.
 */
Code.checkAllGeneratorFunctionsDefined = function(generator) {
  var blocks = Code.workspace.getAllBlocks();
  var missingBlockGenerators = [];
  for (var i = 0; i < blocks.length; i++) {
    var blockType = blocks[i].type;
    if (!generator[blockType]) {
      if (missingBlockGenerators.indexOf(blockType) === -1) {
        missingBlockGenerators.push(blockType);
      }
    }
  }

  var valid = missingBlockGenerators.length == 0;
  if (!valid) {
    var msg = 'The generator code for the following blocks not specified for '
        + generator.name_ + ':\n - ' + missingBlockGenerators.join('\n - ');
    Blockly.alert(msg);  // Assuming synchronous. No callback.
  }
  return valid;
};

/**
 * Initialize Blockly.  Called on page load.
 */
Code.init = function() {
  Code.initLanguage();
  var rtl = Code.isRtl();
  var container = document.getElementById('content_area');
  var onresize = function(e) {
    var bBox = Code.getBBox_(container);
    for (var i = 0; i < Code.TABS_.length; i++) {
      var el = document.getElementById('content_' + Code.TABS_[i]);
      el.style.top = bBox.y + 'px';
      el.style.left = bBox.x + 'px';
      // Height and width need to be set, read back, then set again to
      // compensate for scrollbars.
      el.style.height = bBox.height + 'px';
      el.style.height = (2 * bBox.height - el.offsetHeight) + 'px';
      el.style.width = bBox.width + 'px';
      el.style.width = (2 * bBox.width - el.offsetWidth) + 'px';
    }
    Code.workspace.options.oneBasedIndex=0;
    // Make the 'Blocks' tab line up with the toolbox.
    if (Code.workspace && Code.workspace.toolbox_.width) {
      document.getElementById('tab_blocks').style.minWidth =
          (Code.workspace.toolbox_.width - 38) + 'px';
          // Account for the 19 pixel margin and on each side.
    }
  };
  window.addEventListener('resize', onresize, false);

  // The toolbox XML specifies each category name using Blockly's messaging
  // format (eg. `<category name="%{BKY_CATLOGIC}">`).
  // These message keys need to be defined in `Blockly.Msg` in order to
  // be decoded by the library. Therefore, we'll use the `MSG` dictionary that's
  // been defined for each language to import each category name message
  // into `Blockly.Msg`.
  // TODO: Clean up the message files so this is done explicitly instead of
  // through this for-loop.
  for (var messageKey in MSG) {
    if (messageKey.indexOf('cat') == 0) {
      Blockly.Msg[messageKey.toUpperCase()] = MSG[messageKey];
    }
  }

  // Construct the toolbox XML, replacing translated variable names.
  var toolboxText = document.getElementById('toolbox').outerHTML;
  toolboxText = toolboxText.replace(/(^|[^%]){(\w+)}/g,
      function(m, p1, p2) {return p1 + MSG[p2];});
  var toolboxXml = Blockly.Xml.textToDom(toolboxText);
  Code.workspace = Blockly.inject('content_blocks',
      {grid:
          {spacing: 25,
           length: 3,
           colour: '#ccc',
           snap: true},
       media: '../../media/',
       rtl: rtl,
       toolbox: toolboxXml,
       zoom:
           {controls: true,
            wheel: true}
      });

  // Add to reserved word list: Local variables in execution environment (runJS)
  // and the infinite loop detection function.
  //Blockly.JavaScript.addReservedWords('code,timeouts,checkTimeout');

  Code.loadBlocks('');
  
  // all the custom functions in wattmon
  Code.loadBlockTypes();

  if ('BlocklyStorage' in window) {
    // Hook a save function onto unload.
    BlocklyStorage.backupOnUnload(Code.workspace);
  }

  Code.tabClick(Code.selected);

  Code.bindClick('trashButton',
      function() {Code.discard(); Code.renderContent();});
  Code.bindClick('genButton', Code.genJS);      
  Code.bindClick('runButton', Code.runJS);
  
  Code.bindClick('optionButton',
      function() { 
          $('#optionsFormTitle').val(val_array[1]);
          console.log("Setting Title to "+val_array[1]);
          console.log("Setting Style to "+val_array[2]);
          console.log("Setting Position to "+val_array[3]);
          $('#optionsFormStyle').val(val_array[2]);
          $('#optionsFormPosition').val(val_array[3]);
          var w=val_array[4];
          if (!w) w=1;
          $('#optionsFormWidth').val(w);
          $('#myModal').modal('show')
          });
  
  Code.bindClick('modal_save',
      function() { 
          val_array[1]=$('#optionsFormTitle').val();
          val_array[2]=$('#optionsFormStyle').val();
          val_array[3]=$('#optionsFormPosition').val();
          val_array[4]=$('#optionsFormWidth').val();
          val_array_changed=1;
          $('#myModal').modal('hide')
          
      });
  // Disable the link button if page isn't backed by App Engine storage.
  var linkButton = document.getElementById('linkButton');
  if ('BlocklyStorage' in window) {
    BlocklyStorage['HTTPREQUEST_ERROR'] = MSG['httpRequestError'];
    BlocklyStorage['LINK_ALERT'] = MSG['linkAlert'];
    BlocklyStorage['HASH_ERROR'] = MSG['hashError'];
    BlocklyStorage['XML_ERROR'] = MSG['xmlError'];
    Code.bindClick(linkButton,
        function() {
            var fn = document.getElementById('f_name').value;
              var re = new RegExp("^(\/)([A-Za-z_\-\s0-9\.\/]+)+\.(cgx)$");
              if (fn.match(re)) {
               // alert("Successful match");
               global_url=fn;
              } else {
                alert("Invalid file name, please correct");
                return;
              }

            BlocklyStorage.link(Code.workspace);
            
            });
  } else if (linkButton) {
    linkButton.className = 'disabled';
  }

  for (var i = 0; i < Code.TABS_.length; i++) {
    var name = Code.TABS_[i];
    Code.bindClick('tab_' + name,
        function(name_) {return function() {Code.tabClick(name_);};}(name));
  }
  onresize();
  Blockly.svgResize(Code.workspace);

  // Lazy-load the syntax-highlighting.
  window.setTimeout(Code.importPrettify, 1);
};

/**
 * Initialize the page language.
 */
Code.initLanguage = function() {
  // Set the HTML's language and direction.
  var rtl = Code.isRtl();
  document.dir = rtl ? 'rtl' : 'ltr';
  document.head.parentElement.setAttribute('lang', Code.LANG);

  // Sort languages alphabetically.
  var languages = [];
  for (var lang in Code.LANGUAGE_NAME) {
    languages.push([Code.LANGUAGE_NAME[lang], lang]);
  }
  var comp = function(a, b) {
    // Sort based on first argument ('English', 'Русский', '简体字', etc).
    if (a[0] > b[0]) return 1;
    if (a[0] < b[0]) return -1;
    return 0;
  };
  languages.sort(comp);
  // Populate the language selection menu.
/*  var languageMenu = document.getElementById('languageMenu');
  languageMenu.options.length = 0;
  for (var i = 0; i < languages.length; i++) {
    var tuple = languages[i];
    var lang = tuple[tuple.length - 1];
    var option = new Option(tuple[0], lang);
    if (lang == Code.LANG) {
      option.selected = true;
    }
    languageMenu.options.add(option);
  }
  languageMenu.addEventListener('change', Code.changeLanguage, true);*/

  // Inject language strings.
  document.title += ' ' + MSG['title'];
  //document.getElementById('title').textContent = MSG['title'];
  //document.getElementById('tab_blocks').textContent = MSG['blocks'];

  document.getElementById('linkButton').title = MSG['linkTooltip'];
  document.getElementById('runButton').title = MSG['runTooltip'];
  document.getElementById('genButton').title = MSG['genTooltip'];
  document.getElementById('trashButton').title = MSG['trashTooltip'];
};

Code.runJS = function() {

    var code = "<?\r\n// Generated automatically by Wattmon from "+global_url+".\r\n// Any changes to this file could be overwritten so edit the original file instead.\r\n\r\n"+Blockly.PHP.workspaceToCode(Code.workspace)+"\r\n?>";
    code=code.replace(/\\\\/mg,"\\");
    var fn = document.getElementById('f_name').value;
    var re = new RegExp("^(\/)([A-Za-z_\-\s0-9\.\/]+)+\.(cgx)$");
    if (fn.match(re)) {
        global_url=fn;
    } else {
        alert("Invalid file name, please correct");
        return;
    }
    fn=fn.substr(0,fn.length-1)+"i";
    //alert(fn);
    $.ajax({
        type: "PUT",
        url: fn,
        data: code,
        dataType: "html"
    }).done(function( msg ) {
        if (msg.substr(0,2)!="OK") alert(msg);
    	document.getElementById("statusmsg").innerHTML="<b>Output file "+fn+" generated successfully.</b>";
    	window.open(fn,"_blank");
    });
};

Code.genJS = function() {

    var code = "<?\r\n// Generated automatically by Wattmon from "+global_url+".\r\n// Any changes to this file could be overwritten so edit the original file instead.\r\n\r\n"+Blockly.PHP.workspaceToCode(Code.workspace)+"\r\n?>";
    code=code.replace(/\\\\/mg,"\\");
    var fn = document.getElementById('f_name').value;
    var re = new RegExp("^(\/)([A-Za-z_\-\s0-9\.\/]+)+\.(cgx)$");
    if (fn.match(re)) {
        global_url=fn;
    } else {
        alert("Invalid file name, please correct");
        return;
    }
    fn=fn.substr(0,fn.length-1)+"i";
    //alert(fn);
    $.ajax({
        type: "PUT",
        url: fn,
        data: code,
        dataType: "html"
    }).done(function( msg ) {
        if (msg.substr(0,2)!="OK") alert(msg);
        document.getElementById("statusmsg").innerHTML="<b>Output file "+fn+" generated successfully.</b>";
        if (hook != $('#save_hook').val() || val_array_changed) {
              var hval=$('#save_hook').val();
              if (hval==0) hval="";
              code=fn+"="+hval;
              if (hval==5) { // home page hook, extra parameters
                code = code +"|"+val_array[1]+"|"+val_array[2]+"|"+val_array[3]+"|"+val_array[4];
              }
              if (fn.length>32) {
                  alert("The length of this hook file is too long to be included in the system.  Try saving the hook file elsewhere or reducing the full path and filename to below 32 characters.");
                  return;
              }
              //alert(code);
              console.log(fn+" "+code);
                $.ajax({
                    type: "POST",
                    url: "/app/configsave.cgi?config=hooks.ini&section=config",
                    data: code,
                    dataType: "html"
                }).done(function( msg ) {
                    hook=$('#save_hook').val();
                    //if (msg.substr(0,2)!="OK") alert(msg);
                    //document.getElementById("statusmsg").innerHTML="<b>Output file "+fn+" generated successfully.</b>";
                });
          }
        
    });
};

/**
 * Discard all blocks from the workspace.
 */
Code.discard = function() {
  var count = Code.workspace.getAllBlocks().length;
  if (count < 2 ||
      window.confirm(Blockly.Msg.DELETE_ALL_BLOCKS.replace('%1', count))) {
    Code.workspace.clear();
    if (window.location.hash) {
      window.location.hash = '';
    }
  }
};

// Load Blockly's language strings.
document.write('<script src="/lib/blockly/code/msg/' + Code.LANG + '.js"></script>\n');
// Load the Code demo's language strings.
document.write('<script src="/lib/blockly/msg/js/' + Code.LANG + '.js"></script>\n');

//window.addEventListener('load', Code.init);

